<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Models\Post;

use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Data\Common\CommonQueries;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Tables\Post\PostSaveTable;
use Erlage\Photogram\Data\Tables\Post\PostCommentTable;
use Erlage\Photogram\Data\Tables\Post\PostUserTagTable;
use Erlage\Photogram\Data\Models\Post\Like\PostLikeModel;
use Erlage\Photogram\Data\Models\Post\Save\PostSaveModel;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagPostTable;
use Erlage\Photogram\Data\Models\Hashtag\Post\HashtagPostModel;
use Erlage\Photogram\Data\Models\Post\Comment\PostCommentModel;
use Erlage\Photogram\Data\Models\Post\Usertag\PostUserTagModel;

final class PostModelHelper
{
    public static function updateCacheLikesCount(PostModel $postModel): void
    {
        $currentCount = (new Query())
            -> from(PostLikeTable::TABLE_NAME)
            -> where(PostLikeTable::LIKED_POST_ID, $postModel -> getId())
            -> count();

        $postModel -> update(array(
            PostTable::CACHE_LIKES_COUNT => $currentCount,
        ));

        $postModel -> save();
    }

    public static function updateCacheCommentsCount(PostModel $postModel): void
    {
        $currentCount = (new Query())
            -> from(PostCommentTable::TABLE_NAME)
            -> where(PostCommentTable::PARENT_POST_ID, $postModel -> getId())
            -> count();

        $postModel -> update(array(
            PostTable::CACHE_COMMENTS_COUNT => $currentCount,
        ));

        $postModel -> save();
    }

    public static function deleteDependencies(PostModel $postModel): void
    {
        // delete child models

        /**
         * @var PostCommentModel[]
         */
        $postCommentModels = CommonQueries::beansWithMatchingPredicates(
            PostCommentTable::getTableName(),
            array(
                PostCommentTable::PARENT_POST_ID => $postModel -> getId(),
            )
        );

        foreach ($postCommentModels as $postCommentModel)
        {
            $postCommentModel -> delete();
        }

        /**
         * @var PostLikeModel[]
         */
        $postLikeModels = CommonQueries::beansWithMatchingPredicates(
            PostLikeTable::getTableName(),
            array(
                PostLikeTable::LIKED_POST_ID => $postModel -> getId(),
            )
        );

        foreach ($postLikeModels as $postLikeModel)
        {
            $postLikeModel -> delete();
        }

        /**
         * @var PostSaveModel[]
         */
        $postSaveModels = CommonQueries::beansWithMatchingPredicates(
            PostSaveTable::getTableName(),
            array(
                PostSaveTable::SAVED_POST_ID => $postModel -> getId(),
            )
        );

        foreach ($postSaveModels as $postSaveModel)
        {
            $postSaveModel -> delete();
        }

        /**
         * @var PostUserTagModel[]
         */
        $postUserTagModels = CommonQueries::beansWithMatchingPredicates(
            PostUserTagTable::getTableName(),
            array(
                PostUserTagTable::TAGGED_IN_POST_ID => $postModel -> getId(),
            )
        );

        foreach ($postUserTagModels as $postUserTagModel)
        {
            $postUserTagModel -> delete();
        }

        // delete side models

        /**
        * @var HashtagPostModel[]
        */
        $hashtagPostModels = CommonQueries::beansWithMatchingPredicates(
            HashtagPostTable::getTableName(),
            array(
                HashtagPostTable::POST_ID => $postModel -> getId(),
            )
        );

        foreach ($hashtagPostModels as $hashtagPostModel)
        {
            $hashtagPostModel -> delete();
        }
    }
}
